'use strict'

entityRegistry['module']['depthToBrightness'] = {
    extendedInfo: {
        displayName: 'Depth to Brightness',
        displayGroup: 'Post Processing',
    },
    init: () => {
        return {
        }
    },
    staticConfig: [
    ],
    dynamicConfig: [
        { paramName: 'enableDepth', displayName: 'Enable Depth', type: 'boolean', defaultValue: true},
        { paramName: 'mode', displayName: 'Mode', type: 'string', defaultValue: 'mult', uiOptions: { options: [{text:'Set', value:'set'}, {text:'Multiply', value:'mult'}] }},
        { paramName: 'minDepth', displayName: 'Min Depth', type: 'float', defaultValue: 0, uiOptions: { scrollScales: [0.1, 1, 10] }},
        { paramName: 'maxDepth', displayName: 'Max Depth', type: 'float', defaultValue: 100, uiOptions: { scrollScales: [0.1, 1, 10] }},
        { paramName: 'minBrightness', displayName: 'Min Brightness', type: 'float', defaultValue: 1, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'maxBrightness', displayName: 'Max Brightness', type: 'float', defaultValue: 0, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'enableAO', displayName: 'Enable AO', type: 'boolean', defaultValue: true},
        { paramName: 'xminAODepth', displayName: 'Min Depth', type: 'float', defaultValue: 0.01, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'xmaxAODepth', displayName: 'Max Depth', type: 'float', defaultValue: 0.1, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'strengthAO', displayName: 'Strength', type: 'float', defaultValue: 0.6, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'enableDOF', displayName: 'Enable DOF', type: 'boolean', defaultValue: true},
        { paramName: 'dofBlurPasses', displayName: 'Blur Passes', type: 'int', defaultValue: 3, uiOptions: { min: 1, max: 10 }},
        { paramName: 'focalPoint', displayName: 'Focal Point', type: 'float', defaultValue: 10, uiOptions: { scrollScales: [0.01, 0.1, 1] }},
        { paramName: 'focalLength', displayName: 'Focal Length', type: 'float', defaultValue: 4, uiOptions: { min: 0, scrollScales: [0.01, 0.1, 1] }},
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                mode,
                enableDepth,
                minDepth,
                maxDepth,
                minBrightness,
                maxBrightness,
                enableAO,
                xminAODepth,
                xmaxAODepth,
                strengthAO,
                enableDOF,
                dofBlurPasses,
                focalPoint,
                focalLength,
            } = { ...config }

            const rangeDepth = maxDepth - minDepth

            const depthBuffer = renderer.getCurrentBuffer('depth')
            const depthBufferData = depthBuffer.data
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')
            const brightnessBufferData = brightnessBuffer.data
            const width = depthBuffer.width
            const height = depthBuffer.height
            const halfWidth = width / 2
            const halfHeight = height / 2

            // Depth darken
            if (enableDepth) {
                if (mode === 'set') {
                    for (let i = 0; i < depthBuffer.pixelCount; ++i) {
                        let v = depthBufferData[i]
                        v = clamp((v - minDepth) / rangeDepth, 0, 1)
                        brightnessBufferData[i] = lerp(minBrightness, maxBrightness, v**2)
                    }
                } else if (mode === 'mult') {
                    for (let i = 0; i < depthBuffer.pixelCount; ++i) {
                        let v = depthBufferData[i]
                        v = clamp((v - minDepth) / rangeDepth, 0, 1)
                        brightnessBufferData[i] *= lerp(minBrightness, maxBrightness, v**2)
                    }
                }
            }

            // SSAO
            if (enableAO) {
                const minAODepth = -xminAODepth
                const maxAODepth = -xmaxAODepth
                for (let x = 0; x < width; ++x) {
                    let index = x + 3 * width
                    for (let y = 3; y < height - 3; ++y) {
                        const dp3 = depthBufferData[index - width * 3]
                        const dp2 = depthBufferData[index - width * 2]
                        const dp1 = depthBufferData[index - width * 1]
                        const d0 = depthBufferData[index]
                        const dm1 = depthBufferData[index + width * 1]
                        const dm2 = depthBufferData[index + width * 2]
                        const dm3 = depthBufferData[index + width * 3]
    
                        let v = 1
                        let d
    
                        d = (dm3 - d0) - (d0 - dp3); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO/3
                        d = (dm2 - d0) - (d0 - dp2); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO/2
                        d = (dm1 - d0) - (d0 - dp1); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO

                        if (v !== 1) {
                            v = Math.max(v,0)
                            brightnessBufferData[index] *= v
                        }
    
                        index += width
                    }
                }
                for (let y = 0; y < height; ++y) {
                    let index = y * width + 3
                    for (let x = 3; x < width - 3; ++x) {
                        const dp3 = depthBufferData[index - 3]
                        const dp2 = depthBufferData[index - 2]
                        const dp1 = depthBufferData[index - 1]
                        const d0 = depthBufferData[index]
                        const dm1 = depthBufferData[index + 1]
                        const dm2 = depthBufferData[index + 2]
                        const dm3 = depthBufferData[index + 3]
    
                        let v = 1
                        let d
    
                        d = (dm3 - d0) - (d0 - dp3); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO/3
                        d = (dm2 - d0) - (d0 - dp2); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO/2
                        d = (dm1 - d0) - (d0 - dp1); if (d > maxAODepth && d < minAODepth) v *= 1+(maxAODepth-d)*strengthAO

                        if (v !== 1) {
                            v = Math.max(v, 0)
//                            v = Math.min(v, 1)
                            brightnessBufferData[index] *= v
                        }

                        index += 1
                    }
                }
            }

            // DOF
            if (enableDOF) {
                let dst1Id = renderer.lockBuffer('brightness')
                let dst2Id = renderer.lockBuffer('brightness')
                let dst1 = renderer.getBufferFromId('brightness', dst1Id).data
                let dst2 = renderer.getBufferFromId('brightness', dst2Id).data
                let writeIndex = 0
                for (let y = 0; y < height; y+=2) {
                    for (let x = 0; x < width; x+=2) {
                        const v = (brightnessBufferData[(y+0)*width + x+0] +
                                   brightnessBufferData[(y+0)*width + x+0] +
                                   brightnessBufferData[(y+1)*width + x+1] +
                                   brightnessBufferData[(y+1)*width + x+1]) / 4
                        dst1[writeIndex++] = v
                    }
                }
                for (let blurPass = 0; blurPass < dofBlurPasses; ++blurPass) {
                    {
                        let index = 0
                        for (let x = 0; x < halfWidth; ++x) dst2[index] = dst1[index++]
                        for (let y = 1; y < halfHeight-1; ++y) {
                            dst2[index] = dst1[index++]
                            for (let x = 1; x < halfWidth-1; ++x) {
                                const v = (dst1[index-1] +
                                        dst1[index+1] +
                                        dst1[index] +
                                        dst1[index-halfWidth] +
                                        dst1[index+halfWidth]) / 5
                                dst2[index++] = v
                            }
                            dst2[index] = dst1[index++]
                        }
                        for (let x = 0; x < halfWidth; ++x) dst2[index] = dst1[index++]
                    }
                    {
                        let index = 0
                        for (let x = 0; x < halfWidth; ++x) dst1[index] = dst2[index++]
                        for (let y = 1; y < halfHeight-1; ++y) {
                            dst1[index] = dst2[index++]
                            for (let x = 1; x < halfWidth-1; ++x) {
                                const v = (dst2[index-1] +
                                        dst2[index+1] +
                                        dst2[index] +
                                        dst2[index-halfWidth] +
                                        dst2[index+halfWidth]) / 5
                                dst1[index++] = v
                            }
                            dst1[index] = dst2[index++]
                        }
                        for (let x = 0; x < halfWidth; ++x) dst1[index] = dst2[index++]
                    }
                }
                const iFocalLength = 1/focalLength
                const src = dst1
                const dst = brightnessBufferData
                let index = 0
                for (let y = 0; y < height; ++y) {
                    for (let x = 0; x < width; ++x) {
                        const depth = clamp(Math.abs((depthBufferData[index]-focalPoint) * iFocalLength), 0, 1)
                        dst[index] = Math.max(lerp(dst[index], src[(y>>1)*halfWidth+(x>>1)], depth**2), 0)
                        index++
                    }
                }
                renderer.unlockBuffer('brightness', dst2Id)
                renderer.unlockBuffer('brightness', dst1Id)
            }
        }
    }
}
